/**
 * \file pappsomspp/processing/cbor/psm/evalscan/psmspecglob.cpp
 * \date 19/07/2025
 * \author Olivier Langella
 * \brief compute specglob alignment on scan's PSM
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "psmspecpeptidoms.h"

#include "../../../../processing/filters/filterchargedeconvolution.h"
#include "../../../../processing/filters/filterresample.h"
#include "../../../../processing/filters/filterpass.h"
#include "../../../../peptide/peptideproformaparser.h"
#include "../../../../pappsoexception.h"
#include "../../../../processing/specpeptidoms/semiglobalalignment.h"
#include "../../../../processing/specpeptidoms/spomsspectrum.h"

pappso::cbor::psm::PsmSpecPeptidOms::PsmSpecPeptidOms(std::size_t buffer_scan_size,
                                                      pappso::cbor::CborStreamWriter *cbor_output_p,
                                                      const QJsonObject &parameters)
  : PsmFileScanProcessAndCopy(buffer_scan_size, cbor_output_p, "SpecPeptidOms")
{
  m_specpeptidomsParameters = parameters;

  if(parameters.value("fragment_tolerance_unit").toString() == "dalton")
    {
      m_fragmentTolerance = pappso::PrecisionFactory::getDaltonInstance(
        parameters.value("fragment_tolerance").toDouble());
    }
  else if(parameters.value("fragment_tolerance_unit").toString() == "ppm")
    {
      m_fragmentTolerance =
        pappso::PrecisionFactory::getPpmInstance(parameters.value("fragment_tolerance").toDouble());
    }

  QJsonObject spectrum_param = parameters.value("spectrum").toObject();

  m_minimumMz    = spectrum_param.value("minimum_mz").toDouble();
  m_nMostIntense = spectrum_param.value("n_most_intense").toInt();
  m_deisotope    = spectrum_param.value("deisotope").toBool();

  m_aaCode.addAaModification('C', AaModification::getInstance("MOD:00397"));
}

pappso::cbor::psm::PsmSpecPeptidOms::~PsmSpecPeptidOms()
{
}

void
pappso::cbor::psm::PsmSpecPeptidOms::filterMassSpectrum(pappso::MassSpectrum &mass_spectrum) const
{
  if(m_deisotope)
    pappso::FilterChargeDeconvolution(m_fragmentTolerance).filter(mass_spectrum);
  pappso::FilterResampleKeepGreater(m_minimumMz).filter(mass_spectrum);
  pappso::FilterGreatestY(m_nMostIntense).filter(mass_spectrum);
}

void
pappso::cbor::psm::PsmSpecPeptidOms::parameterMapReady(pappso::UiMonitorInterface &monitor)
{
}

pappso::cbor::psm::CborScanMapBase *
pappso::cbor::psm::PsmSpecPeptidOms::newCborScanMap()
{
  return new PsmSpecPeptidOmsScan(*this, m_fragmentTolerance);
}

const pappso::AaCode &
pappso::cbor::psm::PsmSpecPeptidOms::getAaCode() const
{
  return m_aaCode;
}


pappso::cbor::psm::PsmSpecPeptidOmsScan::PsmSpecPeptidOmsScan(
  const pappso::cbor::psm::PsmSpecPeptidOms &psm_specpeptidoms,
  pappso::PrecisionPtr fragment_tolerance)
  : CborScanMapBase(psm_specpeptidoms)
{
  m_fragmentTolerance = fragment_tolerance;
  mp_psmSpecPeptidOms = &psm_specpeptidoms;
}

pappso::cbor::psm::PsmSpecPeptidOmsScan::~PsmSpecPeptidOmsScan()
{
}

void
pappso::cbor::psm::PsmSpecPeptidOmsScan::process()
{
  // qDebug() << keys();
  if(!keys().contains("id"))
    {
      throw pappso::PappsoException(QObject::tr("missing scan id"));
    }
  if(keys().contains("psm_list"))
    {
      QualifiedMassSpectrumSPtr qualified_mass_spectrum = getCurrentQualifiedMassSpectrumSPtr();


      mp_psmSpecPeptidOms->filterMassSpectrum(
        *(qualified_mass_spectrum.get()->getMassSpectrumSPtr().get()));

      QCborArray new_psm_arr;
      for(QCborValue cbor_psm : value("psm_list").toArray())
        {
          QCborMap cbor_psm_map = cbor_psm.toMap();

          if(!cbor_psm_map.keys().contains("proforma"))
            {
              throw pappso::PappsoException(
                QObject::tr("missing proforma in psm %1").arg(cbor_psm_map.keys().size()));
            }
          QCborMap cbor_psm_eval_specglob;
          pappso::PeptideSp peptide_sp =
            pappso::PeptideProFormaParser::parseString(cbor_psm_map.value("proforma").toString());


          pappso::specpeptidoms::SpOMSSpectrumCsp experimental_spectrum =
            std::make_shared<pappso::specpeptidoms::SpOMSSpectrum>(
              *qualified_mass_spectrum.get(),
              m_fragmentTolerance,
              mp_psmSpecPeptidOms->getAaCode());
          /*
                    pappso::specglob::SpectralAlignment
             spectral_alignment(pappso::specglob::ScoreValues(), m_fragmentTolerance);

                    spectral_alignment.align(peptide_spectra, experimental_spectrum);


                    cbor_psm_eval_specglob.insert(QString("max_score"),
             spectral_alignment.getMaxScore());


                    if(spectral_alignment.getMaxScore() > 0)
                      {
                        pappso::specglob::PeptideModel peptide_model =
             spectral_alignment.buildPeptideModel();

                        // peptide_model.eliminateComplementaryDelta(m_fragmentTolerance);
                        // peptide_model.matchExperimentalPeaks(m_fragmentTolerance);


                        pappso::specglob::PostTreatment post_treatment(m_fragmentTolerance,
             peptide_model); if(m_checkMutations) post_treatment.findReplaceMutations();
                        peptide_model = post_treatment.getBetterPeptideModel();

                        cbor_psm_eval_specglob.insert(QString("count"),
                                                      (qint64)peptide_model.getCountSharedPeaks());

                        cbor_psm_eval_specglob.insert(QString("proforma"),
             peptide_model.toProForma()); cbor_psm_eval_specglob.insert(QString("bracket"),
             peptide_model.toString());

                        // peptide_model.checkForMutations()
                      }


                    QCborMap psm_eval = cbor_psm_map.value("eval").toMap();
                    psm_eval.remove(QString("specglob"));
                    psm_eval.insert(QString("specglob"), cbor_psm_eval_specglob);
                    cbor_psm_map.remove(QString("eval"));
                    cbor_psm_map.insert(QString("eval"), psm_eval);

                    new_psm_arr.push_back(cbor_psm_map);
                    */
        }

      // insert(QString("psm_list"), new_psm_arr);
      remove(QString("psm_list"));
      insert(QString("psm_list"), new_psm_arr);
    }
}
