/* This file is part of the KDE libraries
    SPDX-FileCopyrightText: 2006, 2007 Andreas Hartmetz (ahartmetz@gmail.com)
    SPDX-FileCopyrightText: 2008 Michael Jansen <kde@michael-jansen.biz>
    SPDX-FileCopyrightText: 2008 Alexander Dymo <adymo@kdevelop.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

#ifndef KISSHORTCUTSDIALOG_P_H
#define KISSHORTCUTSDIALOG_P_H

#include "KisShortcutsEditor.h"
#include "kkeysequencewidget.h"
#include "KisShortcutsDialog.h"

#include <kextendableitemdelegate.h>
#include <klocalizedstring.h>
#include <kmessagebox.h>

#include <QKeySequence>
#include <QMetaType>
#include <QModelIndex>
#include <QList>
#include <QCollator>
#include <QHBoxLayout>

class QLabel;
class QTreeWidget;
class QTreeWidgetItem;
class QRadioButton;
class QAction;
class KisKActionCollection;
class QPushButton;
class QComboBox;
class KisShortcutsDialog;
class KisKShortcutSchemesEditor;
class QAction;


enum ColumnDesignation {
    Name = 0,
    LocalPrimary,
    LocalAlternate,
    Id
};

// XXX: Hmm
enum MyRoles {
    ShortcutRole = Qt::UserRole,
    DefaultShortcutRole,
    ObjectRole
};


/**
 * Type used for QTreeWidgetItems
 *
 * @internal
 */
enum ItemTypes {
    NonActionItem = 0,
    ActionItem = 1
};

// Return the first item of the list, if it exists
QKeySequence primarySequence(const QList<QKeySequence> &sequences);

// Return the second item of the list, if it exists
QKeySequence alternateSequence(const QList<QKeySequence> &sequences);



class KisShortcutsDialog::KisShortcutsDialogPrivate
{
public:
    KisShortcutsDialogPrivate(KisShortcutsDialog *q);
    void changeShortcutScheme(const QString &scheme);
    void undo();
    void save();

    QHash<QString, KisKActionCollection *> m_collections;
    KisShortcutsDialog *q;
    KisShortcutsEditor *m_shortcutsEditor {0};
    KisKShortcutSchemesEditor *m_schemeEditor{0};
};


/**
 * Mixes the KisKShortcutWidget into the treeview used by KisShortcutsEditor. When
 * selecting an shortcut it changes the display from "CTRL-W" to the Widget.
 *
 * @bug That delegate uses KExtendableItemDelegate. That means a cell can be
 * expanded. When selected a cell is replaced by a KisShortcutsEditor. When
 * painting the widget KExtendableItemDelegate reparents the widget to the
 * viewport of the itemview it belongs to. The widget is destroyed when the user
 * selects another shortcut or explicitly issues a contractItem event. But when
 * the user clears the model the delegate misses that event and doesn't delete
 * the KShortcutseditor. And remains as a visible artifact in your treeview.
 * Additionally when closing your application you get an assertion failure from
 * KExtendableItemDelegate.
 *
 * @internal
 */
class KisShortcutsEditorDelegate : public KExtendableItemDelegate
{
    Q_OBJECT
public:
    KisShortcutsEditorDelegate(QTreeWidget *parent, bool allowLetterShortcuts);
    //reimplemented to have some extra height
    QSize sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const override;

    /**
     * Set a list of action collections to check against for conflicting
     * shortcuts.
     *
     * @see KisKKeySequenceWidget::setCheckActionCollections
     */
    void setCheckActionCollections(const QList<KisKActionCollection *> checkActionCollections);
    bool eventFilter(QObject *, QEvent *) override;
private:
    mutable QPersistentModelIndex m_editingIndex;
    bool m_allowLetterShortcuts;
    QWidget *m_editor;

    //! List of actionCollections to check for conflicts.
    QList<KisKActionCollection *> m_checkActionCollections;


Q_SIGNALS:
  void shortcutChanged(QVariant, const QModelIndex &);

public Q_SLOTS:
  void hiddenBySearchLine(QTreeWidgetItem *, bool);

private Q_SLOTS:
    void itemActivated(QModelIndex index);

    /**
     * When the user collapses a hole subtree of shortcuts then remove eventually
     * extended items. Else we get that artifact bug. See above.
     */
    void itemCollapsed(QModelIndex index);

    /**
     * If the user allowed stealing a shortcut we want to be able to undo
     * that.
     */
    void stealShortcut(const QKeySequence &seq, QAction *action);

    void keySequenceChanged(const QKeySequence &);

};


/**
 * Edit a shortcut. This widget is displayed when a user clicks on a shortcut
 * from the list. It contains radio buttons choosing between default and custom
 * shortcuts, and a button to configure the custom shortcut.
 *
 * @see KisShortcutsEditorDelegate::itemActivated
 * @see KisShortcutWidget.cpp
 *
 * @internal
 */
class ShortcutEditWidget : public QWidget
{
    Q_OBJECT
public:
    ShortcutEditWidget(QWidget *viewport, const QKeySequence &defaultSeq, const QKeySequence &activeSeq,
                       bool allowLetterShortcuts);

    //! @see KisKKeySequenceWidget::setCheckActionCollections()
    void setCheckActionCollections(const QList<KisKActionCollection *> checkActionCollections);

    //@{
    //! @see KisKKeySequenceWidget::checkAgainstStandardShortcuts()
    KisKKeySequenceWidget::ShortcutTypes checkForConflictsAgainst() const;
    void setCheckForConflictsAgainst(KisKKeySequenceWidget::ShortcutTypes);
    //@}

    //@{
    //! @see KisKKeySequenceWidget::checkAgainstStandardShortcuts()
    bool multiKeyShortcutsAllowed() const;
    void setMultiKeyShortcutsAllowed(bool);
    //@}

    //! @see KisKKeySequenceWidget::setComponentName
    void setComponentName(const QString componentName);

    void setAction(QObject *action);
    void paintEvent(QPaintEvent *pe) override;


Q_SIGNALS:
    //! Emitted when the key sequence is changed.
    void keySequenceChanged(const QKeySequence &);

    //! @see KisKKeySequenceWidget::stealShortcut()
    void stealShortcut(const QKeySequence &seq, QAction *action);


public Q_SLOTS:
  //! Set the displayed sequences
  void setKeySequence(const QKeySequence &activeSeq);

private Q_SLOTS:
    void defaultToggled(bool);
    void setCustom(const QKeySequence &);



private:
    QLabel *m_defaultLabel;
    QKeySequence m_defaultKeySequence;
    QRadioButton *m_defaultRadio;
    QRadioButton *m_customRadio;
    KisKKeySequenceWidget *m_customEditor;
    bool m_isUpdating;
    QObject *m_action;
};

#endif /* KISSHORTCUTSDIALOG_P_H */
